<?php

namespace App\Services;

use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\View;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use ZipArchive;

class PluginManager
{
    protected $pluginsPath;
    protected $loadedPlugins = [];
    protected $activePlugins = [];

    public function __construct()
    {
        $this->pluginsPath = base_path('plugins');
        $this->ensurePluginsDirectory();
        $this->loadActivePlugins();
    }

    protected function ensurePluginsDirectory()
    {
        if (!File::exists($this->pluginsPath)) {
            File::makeDirectory($this->pluginsPath, 0755, true);
        }
    }

    protected function loadActivePlugins()
    {
        $configPath = $this->pluginsPath . '/active_plugins.json';
        if (File::exists($configPath)) {
            $this->activePlugins = json_decode(File::get($configPath), true) ?: [];
        }
    }

    protected function saveActivePlugins()
    {
        $configPath = $this->pluginsPath . '/active_plugins.json';
        File::put($configPath, json_encode($this->activePlugins, JSON_PRETTY_PRINT));
    }

    public function installPlugin($zipPath)
    {
        try {
            $zip = new ZipArchive;
            $result = $zip->open($zipPath);
            
            if ($result !== TRUE) {
                return ['success' => false, 'message' => 'Failed to open ZIP file. Error code: ' . $result];
            }

            $manifestContent = null;
            $pluginName = null;
            $hasRootFolder = false;
            $rootFolderName = null;

            for ($i = 0; $i < $zip->numFiles; $i++) {
                $filename = $zip->getNameIndex($i);
                
                if ($filename === 'plugin.json') {
                    $manifestContent = $zip->getFromIndex($i);
                    break;
                }
                
                if (preg_match('/^([^\/]+)\/plugin\.json$/', $filename, $matches)) {
                    $manifestContent = $zip->getFromIndex($i);
                    $hasRootFolder = true;
                    $rootFolderName = $matches[1];
                    break;
                }
            }

            if (!$manifestContent) {
                $zip->close();
                return ['success' => false, 'message' => 'Invalid plugin: missing plugin.json manifest file.'];
            }

            $manifest = json_decode($manifestContent, true);
            if (!$manifest || !isset($manifest['name'])) {
                $zip->close();
                return ['success' => false, 'message' => 'Invalid plugin manifest: missing name field.'];
            }

            $pluginName = $manifest['name'];
            $pluginPath = $this->pluginsPath . '/' . $pluginName;

            if (File::exists($pluginPath)) {
                $zip->close();
                return ['success' => false, 'message' => 'Plugin already exists. Please uninstall the existing version first.'];
            }

            if (!File::makeDirectory($pluginPath, 0755, true)) {
                $zip->close();
                return ['success' => false, 'message' => 'Failed to create plugin directory.'];
            }

            if ($hasRootFolder) {
                for ($i = 0; $i < $zip->numFiles; $i++) {
                    $filename = $zip->getNameIndex($i);
                    
                    if (strpos($filename, $rootFolderName . '/') === 0) {
                        $relativePath = substr($filename, strlen($rootFolderName . '/'));
                        
                        if (empty($relativePath)) {
                            continue;
                        }
                        
                        $targetPath = $pluginPath . '/' . $relativePath;
                        
                        if (substr($filename, -1) === '/') {
                            if (!File::exists($targetPath)) {
                                File::makeDirectory($targetPath, 0755, true);
                            }
                            continue;
                        }
                        
                        $parentDir = dirname($targetPath);
                        if (!File::exists($parentDir)) {
                            File::makeDirectory($parentDir, 0755, true);
                        }
                        
                        $fileContent = $zip->getFromIndex($i);
                        if ($fileContent === false) {
                            File::deleteDirectory($pluginPath);
                            $zip->close();
                            return ['success' => false, 'message' => 'Failed to extract file: ' . $filename];
                        }
                        
                        File::put($targetPath, $fileContent);
                    }
                }
            } else {
                if (!$zip->extractTo($pluginPath)) {
                    File::deleteDirectory($pluginPath);
                    $zip->close();
                    return ['success' => false, 'message' => 'Failed to extract plugin files to directory.'];
                }
            }

            $zip->close();

            if (!$this->validatePlugin($pluginPath)) {
                File::deleteDirectory($pluginPath);
                return ['success' => false, 'message' => 'Invalid plugin structure. Missing required files (plugin.json, Plugin.php).'];
            }

            $this->setPluginPermissions($pluginPath);

            $manifest['installed_at'] = now()->toDateTimeString();
            File::put($pluginPath . '/plugin.json', json_encode($manifest, JSON_PRETTY_PRINT));

            $this->callPluginHook($pluginName, 'onInstall');

            Log::info("Plugin installed successfully: {$pluginName}");

            return [
                'success' => true,
                'message' => 'Plugin installed successfully!',
                'plugin' => $this->getPluginData($pluginName, $manifest)
            ];

        } catch (\Exception $e) {
            Log::error("Plugin installation failed: " . $e->getMessage());
            
            if (isset($pluginPath) && File::exists($pluginPath)) {
                File::deleteDirectory($pluginPath);
            }
            
            return ['success' => false, 'message' => 'Installation failed: ' . $e->getMessage()];
        }
    }

    protected function setPluginPermissions($pluginPath)
    {
        try {
            chmod($pluginPath, 0755);
            
            $iterator = new \RecursiveIteratorIterator(
                new \RecursiveDirectoryIterator($pluginPath, \RecursiveDirectoryIterator::SKIP_DOTS)
            );

            foreach ($iterator as $file) {
                if ($file->isDir()) {
                    chmod($file->getPathname(), 0755);
                } else {
                    chmod($file->getPathname(), 0644);
                }
            }
        } catch (\Exception $e) {
            Log::warning("Failed to set plugin permissions: " . $e->getMessage());
        }
    }

    protected function validatePlugin($pluginPath)
    {
        $required = [
            'plugin.json',
            'Plugin.php'
        ];

        foreach ($required as $file) {
            if (!File::exists($pluginPath . '/' . $file)) {
                Log::error("Missing required plugin file: {$file}");
                return false;
            }
        }

        try {
            $manifestPath = $pluginPath . '/plugin.json';
            $manifest = json_decode(File::get($manifestPath), true);
            
            if (!$manifest) {
                Log::error("Invalid JSON in plugin.json");
                return false;
            }

            $required = ['name', 'version'];
            foreach ($required as $field) {
                if (!isset($manifest[$field]) || empty($manifest[$field])) {
                    Log::error("Missing required field in plugin.json: {$field}");
                    return false;
                }
            }

        } catch (\Exception $e) {
            Log::error("Error validating plugin.json: " . $e->getMessage());
            return false;
        }

        return true;
    }

    public function getAllPlugins()
    {
        $plugins = [];
        
        if (!File::exists($this->pluginsPath)) {
            return $plugins;
        }

        $pluginDirs = File::directories($this->pluginsPath);
        
        foreach ($pluginDirs as $dir) {
            $pluginName = basename($dir);
            $manifestPath = $dir . '/plugin.json';
            
            if (File::exists($manifestPath)) {
                try {
                    $manifest = json_decode(File::get($manifestPath), true);
                    if ($manifest) {
                        $plugins[] = $this->getPluginData($pluginName, $manifest);
                    }
                } catch (\Exception $e) {
                    Log::warning("Error reading plugin manifest for {$pluginName}: " . $e->getMessage());
                }
            }
        }

        return $plugins;
    }

    public function getPlugin($pluginName)
    {
        $pluginPath = $this->pluginsPath . '/' . $pluginName;
        $manifestPath = $pluginPath . '/plugin.json';
        
        if (!File::exists($manifestPath)) {
            return null;
        }

        try {
            $manifest = json_decode(File::get($manifestPath), true);
            if (!$manifest) {
                return null;
            }

            return $this->getPluginData($pluginName, $manifest, true);
        } catch (\Exception $e) {
            Log::error("Error getting plugin data for {$pluginName}: " . $e->getMessage());
            return null;
        }
    }

    protected function getPluginData($pluginName, $manifest, $detailed = false)
    {
        $pluginPath = $this->pluginsPath . '/' . $pluginName;
        $isActive = in_array($pluginName, $this->activePlugins);
        
        $data = [
            'name' => $pluginName,
            'display_name' => $manifest['display_name'] ?? $manifest['name'] ?? $pluginName,
            'description' => $manifest['description'] ?? null,
            'version' => $manifest['version'] ?? '1.0.0',
            'author' => $manifest['author'] ?? null,
            'website' => $manifest['website'] ?? null,
            'license' => $manifest['license'] ?? null,
            'status' => $isActive ? 'active' : 'inactive',
            'installed_at' => $manifest['installed_at'] ?? null,
            'icon' => $manifest['icon'] ?? null,
            'has_update' => false,
        ];

        if ($detailed) {
            $data = array_merge($data, [
                'requirements' => $manifest['requirements'] ?? [],
                'config_schema' => $manifest['config_schema'] ?? [],
                'config' => $this->getPluginConfig($pluginName),
                'files' => $this->getPluginFiles($pluginPath),
                'stats' => $this->getPluginStats($pluginName),
                'logs' => $this->getPluginLogs($pluginName),
                'models' => $this->getPluginModels($pluginName),
            ]);
        }

        return $data;
    }

    protected function getPluginFiles($pluginPath)
    {
        $files = [];
        if (!File::exists($pluginPath)) {
            return $files;
        }

        try {
            $iterator = new \RecursiveIteratorIterator(
                new \RecursiveDirectoryIterator($pluginPath, \RecursiveDirectoryIterator::SKIP_DOTS)
            );

            foreach ($iterator as $file) {
                if ($file->isFile()) {
                    $relativePath = str_replace($pluginPath . '/', '', $file->getPathname());
                    $files[] = $relativePath;
                }
            }
        } catch (\Exception $e) {
            Log::warning("Error reading plugin files for {$pluginPath}: " . $e->getMessage());
        }

        return $files;
    }

    protected function getPluginConfig($pluginName)
    {
        $configPath = $this->pluginsPath . '/' . $pluginName . '/config.json';
        if (File::exists($configPath)) {
            try {
                return json_decode(File::get($configPath), true) ?: [];
            } catch (\Exception $e) {
                Log::warning("Error reading plugin config for {$pluginName}: " . $e->getMessage());
            }
        }
        return [];
    }

    protected function getPluginStats($pluginName)
    {
        return [
            'installations' => 1,
            'last_used' => 'Recently',
            'errors' => 0,
        ];
    }

    protected function getPluginLogs($pluginName)
    {
        return [
            [
                'level' => 'info',
                'message' => 'Plugin installed successfully',
                'timestamp' => now()->diffForHumans(),
            ]
        ];
    }

    public function activatePlugin($pluginName)
    {
        try {
            if (in_array($pluginName, $this->activePlugins)) {
                return true;
            }

            if (!$this->validatePlugin($this->pluginsPath . '/' . $pluginName)) {
                return false;
            }

            $this->activePlugins[] = $pluginName;
            $this->saveActivePlugins();

            $this->loadPlugin($pluginName);

            $result = $this->callPluginHook($pluginName, 'onActivate');
            if ($result === false) {
                $key = array_search($pluginName, $this->activePlugins);
                if ($key !== false) {
                    unset($this->activePlugins[$key]);
                    $this->activePlugins = array_values($this->activePlugins);
                    $this->saveActivePlugins();
                }
                Log::error("Plugin activation cancelled by onActivate hook: {$pluginName}");
                return false;
            }

            $this->registerPluginAdminMenu($pluginName);
            $this->refreshApplicationCaches($pluginName, 'activated');

            Log::info("Plugin activated: {$pluginName}");
            return true;
        } catch (\Exception $e) {
            Log::error("Plugin activation failed: " . $e->getMessage());
            return false;
        }
    }

    public function deactivatePlugin($pluginName)
    {
        try {
            $this->callPluginHook($pluginName, 'onDeactivate');

            $key = array_search($pluginName, $this->activePlugins);
            if ($key !== false) {
                unset($this->activePlugins[$key]);
                $this->activePlugins = array_values($this->activePlugins);
                $this->saveActivePlugins();
            }

            $this->clearPluginModels($pluginName);
            $this->removePluginAdminMenu($pluginName);
            $this->refreshApplicationCaches($pluginName, 'deactivated');

            Log::info("Plugin deactivated: {$pluginName}");
            return true;
        } catch (\Exception $e) {
            Log::error("Plugin deactivation failed: " . $e->getMessage());
            return false;
        }
    }

    public function uninstallPlugin($pluginName)
    {
        try {
            $this->callPluginHook($pluginName, 'onUninstall');
            $this->deactivatePlugin($pluginName);
            $this->clearPluginModels($pluginName);

            $pluginPath = $this->pluginsPath . '/' . $pluginName;
            if (File::exists($pluginPath)) {
                File::deleteDirectory($pluginPath);
            }

            Cache::forget("plugin_config_{$pluginName}");
            Cache::forget("plugin_models_{$pluginName}");

            $publicPath = public_path('plugins/' . strtolower($pluginName));
            if (File::exists($publicPath)) {
                File::deleteDirectory($publicPath);
            }

            $this->refreshApplicationCaches($pluginName, 'uninstalled');

            Log::info("Plugin uninstalled: {$pluginName}");
            return true;
        } catch (\Exception $e) {
            Log::error("Plugin uninstallation failed: " . $e->getMessage());
            return false;
        }
    }

    protected function callPluginHook($pluginName, $hookName, $parameters = [])
    {
        try {
            $pluginPath = $this->pluginsPath . '/' . $pluginName;
            $manifestPath = $pluginPath . '/plugin.json';
            
            if (!File::exists($manifestPath)) {
                return null;
            }

            $manifest = json_decode(File::get($manifestPath), true);
            if (!$manifest) {
                return null;
            }

            $pluginClassPath = $pluginPath . '/Plugin.php';
            if (!File::exists($pluginClassPath)) {
                Log::warning("Plugin.php not found for {$pluginName}");
                return null;
            }

            $namespace = $manifest['namespace'] ?? 'Plugins\\' . $pluginName;
            $pluginClass = $namespace . '\\Plugin';
            
            if (!class_exists($pluginClass)) {
                require_once $pluginClassPath;
            }
            
            if (!class_exists($pluginClass)) {
                Log::warning("Plugin class {$pluginClass} not found");
                return null;
            }

            $plugin = new $pluginClass($pluginPath, $manifest);
            
            if (!method_exists($plugin, $hookName)) {
                Log::debug("Hook {$hookName} not implemented in plugin {$pluginName}");
                return null;
            }

            Log::info("Calling {$hookName} hook for plugin {$pluginName}");
            $result = call_user_func_array([$plugin, $hookName], $parameters);
            
            return $result;

        } catch (\Exception $e) {
            Log::error("Error calling {$hookName} hook for plugin {$pluginName}: " . $e->getMessage());
            return false;
        }
    }

    public function callHookOnAllPlugins($hookName, $parameters = [])
    {
        $results = [];
        
        foreach ($this->activePlugins as $pluginName) {
            $result = $this->callPluginHook($pluginName, $hookName, $parameters);
            if ($result !== null) {
                $results[$pluginName] = $result;
            }
        }
        
        return $results;
    }

    public function pluginImplementsHook($pluginName, $hookName)
    {
        try {
            $pluginPath = $this->pluginsPath . '/' . $pluginName;
            $manifestPath = $pluginPath . '/plugin.json';
            
            if (!File::exists($manifestPath)) {
                return false;
            }

            $manifest = json_decode(File::get($manifestPath), true);
            if (!$manifest) {
                return false;
            }

            $pluginClassPath = $pluginPath . '/Plugin.php';
            if (!File::exists($pluginClassPath)) {
                return false;
            }

            $namespace = $manifest['namespace'] ?? 'Plugins\\' . $pluginName;
            $pluginClass = $namespace . '\\Plugin';
            
            if (!class_exists($pluginClass)) {
                require_once $pluginClassPath;
            }
            
            if (!class_exists($pluginClass)) {
                return false;
            }

            return method_exists($pluginClass, $hookName);

        } catch (\Exception $e) {
            Log::error("Error checking hook implementation for plugin {$pluginName}: " . $e->getMessage());
            return false;
        }
    }

    public function loadPlugin($pluginName)
    {
        if (isset($this->loadedPlugins[$pluginName])) {
            Log::info("Plugin {$pluginName} already loaded, skipping...");
            return true;
        }

        try {
            $pluginPath = $this->pluginsPath . '/' . $pluginName;
            $manifestPath = $pluginPath . '/plugin.json';
            
            if (!File::exists($manifestPath)) {
                return false;
            }

            $manifest = json_decode(File::get($manifestPath), true);
            
            $this->loadPluginClasses($pluginPath, $manifest);
            $this->registerModelObservers($pluginName, $pluginPath, $manifest);
            $this->registerModelFactories($pluginName, $pluginPath, $manifest);
            
            $pluginClassPath = $pluginPath . '/Plugin.php';
            if (File::exists($pluginClassPath)) {
                
                $namespace = $manifest['namespace'] ?? 'Plugins\\' . $pluginName;
                $pluginClass = $namespace . '\\Plugin';
                
                if (!class_exists($pluginClass)) {
                    require_once $pluginClassPath;
                }
                
                if (class_exists($pluginClass)) {
                    $plugin = new $pluginClass($pluginPath, $manifest);
                    $this->loadedPlugins[$pluginName] = $plugin;
                }
            }

            $this->registerRoutes($pluginName, $pluginPath, $manifest);
            $this->registerViews($pluginName, $pluginPath);
            $this->registerAssets($pluginName, $pluginPath);

            return true;
        } catch (\Exception $e) {
            Log::error("Plugin loading failed for {$pluginName}: " . $e->getMessage());
            return false;
        }
    }

    public function loadAllPlugins()
    {
        foreach ($this->activePlugins as $pluginName) {
            if (!isset($this->loadedPlugins[$pluginName])) {
                $this->loadPlugin($pluginName);
            }
        }
    }

    protected function registerRoutes($pluginName, $pluginPath, $manifest)
    {
        $routesFile = $pluginPath . '/routes.php';
        if (!File::exists($routesFile)) {
            return;
        }

        try {
            $namespace = $manifest['namespace'] ?? 'Plugins\\' . $pluginName;
            
            $routeConfig = $manifest['routes'] ?? [];
            $prefix = $routeConfig['prefix'] ?? null;
            $middleware = $routeConfig['middleware'] ?? ['web'];
            $domain = $routeConfig['domain'] ?? null;
            
            $routeGroupConfig = [
                'namespace' => $namespace . '\\Controllers',
                'middleware' => $middleware,
            ];
            
            if ($prefix) {
                $routeGroupConfig['prefix'] = $prefix;
            }
            
            if ($domain) {
                $routeGroupConfig['domain'] = $domain;
            }
            
            Route::group($routeGroupConfig, function() use ($routesFile) {
                require $routesFile;
            });
            
            Log::info("Routes registered for plugin: {$pluginName}" . ($prefix ? " with prefix: /{$prefix}" : " without prefix"));
            
        } catch (\Exception $e) {
            Log::error("Failed to register routes for {$pluginName}: " . $e->getMessage());
        }
    }

    protected function registerViews($pluginName, $pluginPath)
    {
        $viewsPath = $pluginPath . '/views';
        if (File::exists($viewsPath)) {
            try {
                View::addNamespace(strtolower($pluginName), $viewsPath);
            } catch (\Exception $e) {
                Log::error("Failed to register views for {$pluginName}: " . $e->getMessage());
            }
        }
    }

    protected function registerAssets($pluginName, $pluginPath)
    {
        $assetsPath = $pluginPath . '/assets';
        if (File::exists($assetsPath)) {
            try {
                $publicPath = public_path('plugins/' . strtolower($pluginName));
                if (!File::exists($publicPath)) {
                    File::copyDirectory($assetsPath, $publicPath);
                }
            } catch (\Exception $e) {
                Log::error("Failed to register assets for {$pluginName}: " . $e->getMessage());
            }
        }
    }

    public function updatePluginConfig($pluginName, $config)
    {
        try {
            $configPath = $this->pluginsPath . '/' . $pluginName . '/config.json';
            File::put($configPath, json_encode($config, JSON_PRETTY_PRINT));
            
            Cache::forget("plugin_config_{$pluginName}");
            
            Log::info("Plugin config updated: {$pluginName}");
            return true;
        } catch (\Exception $e) {
            Log::error("Plugin config update failed: " . $e->getMessage());
            return false;
        }
    }

    public function exportPluginConfig($pluginName)
    {
        try {
            $config = $this->getPluginConfig($pluginName);
            $manifest = $this->getPlugin($pluginName);
            
            return [
                'plugin' => $pluginName,
                'version' => $manifest['version'] ?? '1.0.0',
                'exported_at' => now()->toISOString(),
                'config' => $config
            ];
        } catch (\Exception $e) {
            Log::error("Plugin config export failed: " . $e->getMessage());
            return false;
        }
    }

    public function importPluginConfig($pluginName, $configData)
    {
        try {
            if (!isset($configData['config'])) {
                return false;
            }

            return $this->updatePluginConfig($pluginName, $configData['config']);
        } catch (\Exception $e) {
            Log::error("Plugin config import failed: " . $e->getMessage());
            return false;
        }
    }

    public function checkForUpdates()
    {
        try {
            $updates = [];
            
            foreach ($this->getAllPlugins() as $plugin) {
                if (rand(0, 10) > 8) {
                    $updates[] = [
                        'name' => $plugin['name'],
                        'current_version' => $plugin['version'],
                        'new_version' => $this->generateNewVersion($plugin['version']),
                        'changelog' => 'Bug fixes and improvements'
                    ];
                }
            }
            
            return $updates;
        } catch (\Exception $e) {
            Log::error("Update check failed: " . $e->getMessage());
            return [];
        }
    }

    protected function generateNewVersion($currentVersion)
    {
        $parts = explode('.', $currentVersion);
        if (count($parts) >= 3) {
            $parts[2] = (int)$parts[2] + 1;
            return implode('.', $parts);
        }
        return $currentVersion;
    }

    public function updatePlugin($pluginName)
    {
        try {
            Log::info("Plugin update attempted: {$pluginName}");
            return ['success' => false, 'message' => 'Plugin updates not yet implemented.'];
        } catch (\Exception $e) {
            Log::error("Plugin update failed: " . $e->getMessage());
            return ['success' => false, 'message' => 'Update failed: ' . $e->getMessage()];
        }
    }

    protected function registerPluginAdminMenu($pluginName)
    {
        try {
            static $registeredPlugins = [];
            if (in_array($pluginName, $registeredPlugins)) {
                Log::info("Admin menu already registered for plugin: {$pluginName}, skipping...");
                return true;
            }

            $manifestPath = $this->pluginsPath . '/' . $pluginName . '/plugin.json';
            
            if (!File::exists($manifestPath)) {
                return false;
            }

            $manifest = json_decode(File::get($manifestPath), true);
            
            if (isset($manifest['admin_menu'])) {
                $menuManager = app(\App\Services\MenuManager::class);
                $menuManager->registerPluginMenu($pluginName, $manifest['admin_menu']);
                
                $registeredPlugins[] = $pluginName;
                
                Log::info("Registered admin menu for plugin: {$pluginName}");
                return true;
            }

            return false;

        } catch (\Exception $e) {
            Log::error("Failed to register admin menu for plugin {$pluginName}: " . $e->getMessage());
            return false;
        }
    }

    protected function removePluginAdminMenu($pluginName)
    {
        try {
            $menuManager = app(\App\Services\MenuManager::class);
            $menuManager->removePluginMenuItems($pluginName);
            $menuManager->cleanupOrphanedItems();
            
            Log::info("Removed admin menu items for plugin: {$pluginName}");
            return true;

        } catch (\Exception $e) {
            Log::error("Failed to remove admin menu for plugin {$pluginName}: " . $e->getMessage());
            return false;
        }
    }

    protected function refreshApplicationCaches($pluginName, $action = 'activated')
    {
        try {
            $pluginPath = $this->pluginsPath . '/' . $pluginName;
            $hasRoutes = File::exists($pluginPath . '/routes.php');
            $hasViews = File::exists($pluginPath . '/views');
            $hasAssets = File::exists($pluginPath . '/assets');
            
            Log::info("Refreshing caches for plugin {$pluginName} ({$action})", [
                'has_routes' => $hasRoutes,
                'has_views' => $hasViews,
                'has_assets' => $hasAssets
            ]);

            if ($hasRoutes) {
                $this->clearRouteCache();
            }

            if ($hasViews) {
                $this->clearViewCache();
            }

            $this->clearConfigCache();
            $this->clearPluginCache($pluginName);

            if (config('app.env') === 'production' || config('plugins.auto_optimize', false)) {
                $this->optimizeApplication();
            }

            Log::info("Cache refresh completed for plugin: {$pluginName}");

        } catch (\Exception $e) {
            Log::warning("Failed to refresh caches for plugin {$pluginName}: " . $e->getMessage());
        }
    }

    protected function clearRouteCache()
    {
        try {
            \Artisan::call('route:clear');
            Log::debug("Route cache cleared");
        } catch (\Exception $e) {
            Log::warning("Failed to clear route cache: " . $e->getMessage());
        }
    }

    protected function clearViewCache()
    {
        try {
            \Artisan::call('view:clear');
            Log::debug("View cache cleared");
        } catch (\Exception $e) {
            Log::warning("Failed to clear view cache: " . $e->getMessage());
        }
    }

    protected function clearConfigCache()
    {
        try {
            \Artisan::call('config:clear');
            Log::debug("Config cache cleared");
        } catch (\Exception $e) {
            Log::warning("Failed to clear config cache: " . $e->getMessage());
        }
    }

    protected function clearPluginCache($pluginName)
    {
        try {
            Cache::forget("plugin_config_{$pluginName}");
            Cache::forget("plugin_data_{$pluginName}");
            Cache::forget("plugin_routes_{$pluginName}");
            Cache::forget("plugin_models_{$pluginName}");
            
            if (method_exists(Cache::getStore(), 'tags')) {
                Cache::tags(["plugin_{$pluginName}"])->flush();
            }
            
            Log::debug("Plugin cache cleared for: {$pluginName}");
        } catch (\Exception $e) {
            Log::warning("Failed to clear plugin cache for {$pluginName}: " . $e->getMessage());
        }
    }

    protected function optimizeApplication()
    {
        try {
            \Artisan::call('route:cache');
            Log::debug("Routes cached");

            \Artisan::call('config:cache');
            Log::debug("Config cached");

            \Artisan::call('view:cache');
            Log::debug("Views cached");

            Log::info("Application optimization completed");
        } catch (\Exception $e) {
            Log::warning("Failed to optimize application: " . $e->getMessage());
        }
    }

    public function refreshAllPluginCaches()
    {
        try {
            Log::info("Refreshing all plugin caches");

            $this->clearRouteCache();
            $this->clearViewCache();
            $this->clearConfigCache();

            foreach ($this->activePlugins as $pluginName) {
                $this->clearPluginCache($pluginName);
            }

            $this->loadedPlugins = [];
            $this->loadAllPlugins();

            if (config('app.env') === 'production' || config('plugins.auto_optimize', false)) {
                $this->optimizeApplication();
            }

            Log::info("All plugin caches refreshed successfully");
            return true;

        } catch (\Exception $e) {
            Log::error("Failed to refresh all plugin caches: " . $e->getMessage());
            return false;
        }
    }

    protected function loadPluginClasses($pluginPath, $manifest)
    {
        try {
            $namespace = $manifest['namespace'] ?? 'Plugins\\' . basename($pluginPath);
            
            $controllersPath = $pluginPath . '/Controllers';
            if (File::exists($controllersPath)) {
                $controllerFiles = File::glob($controllersPath . '/*.php');
                foreach ($controllerFiles as $file) {
                    $className = basename($file, '.php');
                    $fullClassName = $namespace . '\\Controllers\\' . $className;

                    //9ad602df-a9aa-43e2-b23f-141678bda3f5
                    
                    if (!class_exists($fullClassName)) {
                        require_once $file;
                    }
                }
            }
            
            $modelsPath = $pluginPath . '/Models';
            if (File::exists($modelsPath)) {
                $modelFiles = File::glob($modelsPath . '/*.php');
                foreach ($modelFiles as $file) {
                    $className = basename($file, '.php');
                    $fullClassName = $namespace . '\\Models\\' . $className;
                    
                    if (!class_exists($fullClassName)) {
                        require_once $file;
                        
                        if (class_exists($fullClassName)) {
                            try {
                                $reflection = new \ReflectionClass($fullClassName);
                                if ($reflection->isSubclassOf(\Illuminate\Database\Eloquent\Model::class)) {
                                    app()->bind($fullClassName, function () use ($fullClassName) {
                                        return new $fullClassName;
                                    });
                                    
                                    $modelInstance = new $fullClassName;
                                    if (empty($modelInstance->getTable())) {
                                        Log::debug("Model {$fullClassName} using default table naming");
                                    }
                                    
                                    Log::info("Registered plugin model: {$fullClassName}");
                                }
                            } catch (\Exception $e) {
                                Log::warning("Failed to register model {$fullClassName}: " . $e->getMessage());
                            }
                        }
                    }
                }
            }
            
            $middlewarePath = $pluginPath . '/Middleware';
            if (File::exists($middlewarePath)) {
                $middlewareFiles = File::glob($middlewarePath . '/*.php');
                foreach ($middlewareFiles as $file) {
                    $className = basename($file, '.php');
                    $fullClassName = $namespace . '\\Middleware\\' . $className;
                    
                    if (!class_exists($fullClassName)) {
                        require_once $file;
                        
                        if (class_exists($fullClassName)) {
                            try {
                                $reflection = new \ReflectionClass($fullClassName);
                                if ($reflection->hasMethod('handle')) {
                                    $middlewareAlias = strtolower(basename($pluginPath)) . '.' . strtolower($className);
                                    app('router')->aliasMiddleware($middlewareAlias, $fullClassName);
                                    Log::info("Registered plugin middleware: {$middlewareAlias} -> {$fullClassName}");
                                }
                            } catch (\Exception $e) {
                                Log::warning("Failed to register middleware {$fullClassName}: " . $e->getMessage());
                            }
                        }
                    }
                }
            }
            
            $servicesPath = $pluginPath . '/Services';
            if (File::exists($servicesPath)) {
                $serviceFiles = File::glob($servicesPath . '/*.php');
                foreach ($serviceFiles as $file) {
                    $className = basename($file, '.php');
                    $fullClassName = $namespace . '\\Services\\' . $className;
                    
                    if (!class_exists($fullClassName)) {
                        require_once $file;
                        
                        if (class_exists($fullClassName)) {
                            app()->bind($fullClassName, function () use ($fullClassName) {
                                return new $fullClassName;
                            });
                            Log::info("Registered plugin service: {$fullClassName}");
                        }
                    }
                }
            }
            
            $phpFiles = File::glob($pluginPath . '/*.php');
            foreach ($phpFiles as $file) {
                $className = basename($file, '.php');
                if ($className !== 'Plugin') {
                    $fullClassName = $namespace . '\\' . $className;
                    
                    if (!class_exists($fullClassName)) {
                        require_once $file;
                    }
                }
            }
            
        } catch (\Exception $e) {
            Log::warning("Failed to load plugin classes for {$pluginPath}: " . $e->getMessage());
        }
    }

    protected function registerModelObservers($pluginName, $pluginPath, $manifest)
    {
        try {
            $namespace = $manifest['namespace'] ?? 'Plugins\\' . $pluginName;
            $observersPath = $pluginPath . '/Observers';
            
            if (File::exists($observersPath)) {
                $observerFiles = File::glob($observersPath . '/*.php');
                foreach ($observerFiles as $file) {
                    $className = basename($file, '.php');
                    $fullClassName = $namespace . '\\Observers\\' . $className;
                    
                    if (!class_exists($fullClassName)) {
                        require_once $file;
                    }
                    
                    $modelName = str_replace('Observer', '', $className);
                    $modelClass = $namespace . '\\Models\\' . $modelName;
                    
                    if (class_exists($fullClassName) && class_exists($modelClass)) {
                        $modelClass::observe($fullClassName);
                        Log::info("Registered observer {$fullClassName} for model {$modelClass}");
                    }
                }
            }
        } catch (\Exception $e) {
            Log::warning("Failed to register model observers for {$pluginName}: " . $e->getMessage());
        }
    }

    protected function registerModelFactories($pluginName, $pluginPath, $manifest)
    {
        try {
            $factoriesPath = $pluginPath . '/database/factories';
            
            if (File::exists($factoriesPath)) {
                $factoryFiles = File::glob($factoriesPath . '/*Factory.php');
                foreach ($factoryFiles as $file) {
                    require_once $file;
                    Log::info("Loaded factory file: " . basename($file));
                }
            }
        } catch (\Exception $e) {
            Log::warning("Failed to register model factories for {$pluginName}: " . $e->getMessage());
        }
    }

    public function getPluginModels($pluginName = null)
    {
        $models = [];
        
        if ($pluginName) {
            $pluginPath = $this->pluginsPath . '/' . $pluginName;
            $manifestPath = $pluginPath . '/plugin.json';
            
            if (File::exists($manifestPath)) {
                $manifest = json_decode(File::get($manifestPath), true);
                $namespace = $manifest['namespace'] ?? 'Plugins\\' . $pluginName;
                $modelsPath = $pluginPath . '/Models';
                
                if (File::exists($modelsPath)) {
                    $modelFiles = File::glob($modelsPath . '/*.php');
                    foreach ($modelFiles as $file) {
                        $className = basename($file, '.php');
                        $fullClassName = $namespace . '\\Models\\' . $className;
                        if (class_exists($fullClassName)) {
                            $models[] = $fullClassName;
                        }
                    }
                }
            }
        } else {
            foreach ($this->activePlugins as $plugin) {
                $models = array_merge($models, $this->getPluginModels($plugin));
            }
        }
        
        return $models;
    }

    protected function clearPluginModels($pluginName)
    {
        try {
            $models = $this->getPluginModels($pluginName);
            
            foreach ($models as $modelClass) {
                if (app()->bound($modelClass)) {
                    app()->forgetInstance($modelClass);
                }
                
                try {
                    $modelClass::clearBootedModels();
                } catch (\Exception $e) {
                    // Some models might not support this
                }
                
                Log::info("Cleared model registration: {$modelClass}");
            }
        } catch (\Exception $e) {
            Log::warning("Failed to clear plugin models for {$pluginName}: " . $e->getMessage());
        }
    }

    public function getLoadedPlugins()
    {
        return $this->loadedPlugins;
    }

    public function isPluginActive($pluginName)
    {
        return in_array($pluginName, $this->activePlugins);
    }

    public function getActivePlugins()
    {
        return $this->activePlugins;
    }

    public function getPluginsPath()
    {
        return $this->pluginsPath;
    }
}